#!/usr/bin/python3

"""
Markdown-Table-Formatter is a minimalistic utility for formatting Markdown
tables. It calculates the maximum column widths to align the content, allowing
you to paste the result directly into a Markdown document.

Version 1.0 - 2025-12-03

Copyright (c) [2025] [Frank ENDRES]

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
of the Software, and to permit persons to whom the Software is furnished to
do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF
OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
"""

from typing import List, Optional
import gi
import sys
import signal


#gettext faker:
# ~ import gettext; _ = gettext.gettext
import locale
messages = {
	'en': { 'Copy': None, 'Paste': None, 'Markdown table or CSV with <Tab> separator:': None, 'Markdown table:': None },
	'fr': { 'Copy': 'Copier', 'Paste': 'Coller', 'Markdown table or CSV with <Tab> separator:': 'Tableau Markdown ou CSV avec séparateur <Tab> :', 'Markdown table:': 'Tableau Markdown :' },
	'de': { 'Copy': 'Kopieren', 'Paste': 'Einfügen', 'Markdown table or CSV with <Tab> separator:': 'Markdown-Tabelle oder CSV mit <Tab>-Separator:', 'Markdown table:': 'Markdown-Tabelle:' },
}
def _(text: str) -> str:
	lang = locale.getlocale()[0][:2]
	return text if lang not in messages.keys() or text not in messages[lang].keys() or messages[lang][text] is None else messages[lang][text]


# GTK3 / 4 compatibility layer -> code is written in GTK4 style
if sys.argv[0].find('-gtk3.') != -1: #GTK3
	gi.require_version('Gtk', '3.0'); from gi.repository import GLib, Gtk, Gio, Gdk
	class CompatApplicationWindow(Gtk.ApplicationWindow): #GTK3
		def set_child(self, child: Gtk.Widget): #renamed method
			for content in self.get_children(): #keep header bar and remove other content
				if not isinstance(content, CompatHeaderBar):
					self.remove(content)
			self.add(child)
		def present(self): #renamed method
			self.show_all()
			super().present()
		def connect(self, event: str, callback):
			if event == 'close-request':
				def close_callback(window, event): #changed event name and callback profile
					callback(window)
				super().connect('delete-event', close_callback)
			else:
				super().connect(event, callback)
	class CompatWindow(Gtk.Window):
		def set_child(self, child: Gtk.Widget): #renamed method
			self.add(child)
		def present(self): #renamed method
			self.show_all()
			super().present()
		def set_default_widget(self, widget: Gtk.Widget):
			widget.set_can_default(True)
			self.set_default(widget)
	class CompatScrolledWindow(Gtk.ScrolledWindow):
		def set_child(self, child: Gtk.Widget): #renamed method
			self.add(child)
	class CompatBox(Gtk.Box):
		def append(self, widget: Gtk.Widget): #renamed method
			self.pack_start(widget, False, False, 0)
	class CompatButton(Gtk.Button):
		def new_from_icon_name(icon_name: str): #changed method profile
			#Gtk.Button.new_from_icon_name(icon_name, Gtk.IconSize.BUTTON) returns a Gtk.Button not a CompatButton
			button = CompatButton()
			button.set_icon_name(icon_name)
			return button
		def set_icon_name(self, icon_name: str):
			image = Gtk.Image.new_from_icon_name(icon_name, Gtk.IconSize.BUTTON)
			self.set_image(image)
	class CompatRadioButton(Gtk.RadioButton):
		def set_group(self, radio): #renamed method
			self.join_group(radio)
	class CompatAboutDialog(Gtk.AboutDialog):
		def set_visible(self, visible: bool): #renamed method
			if visible == True:
				super().run()
				self.destroy()
	class CompatHeaderBar(Gtk.HeaderBar):
		def __init__(self):
			super().__init__()
			self.set_show_close_button(True)
		def set_title_widget(self, label: Gtk.Label):
			self.props.title = label.get_label()
	class CompatPasswordEntry(Gtk.Box):
		def __init__(self):
			super().__init__(orientation = Gtk.Orientation.HORIZONTAL, spacing=10)
			self.password_entry = Gtk.Entry()
			self.password_entry.set_visibility(False)
			self.password_entry.set_activates_default(True)
			self.pack_start(self.password_entry, True, True, 0)
			show_password_button = CompatButton.new_from_icon_name('media-optical-symbolic')
			show_password_button.set_tooltip_text(_('show / hide password'))
			self.pack_start(show_password_button, False, False, 0)
			show_password_button.connect('clicked', lambda bt: self.password_entry.set_visibility(not self.password_entry.get_visibility()))
		def get_text(self) -> str:
			return self.password_entry.get_text()
		def connect(self, event, callback):
			self.password_entry.connect(event, callback)
	class CompatStringListStore(Gtk.ListStore):
		def __init__(self):
			super().__init__(str)
		def append(self, value: str):
			super().append([value])
		def remove_all(self):
			self.clear()
	class CompatDropDown(Gtk.Entry):
		def __init__(self):
			super().__init__()
			self.completion = Gtk.EntryCompletion()
			self.completion.set_text_column(0)
			self.completion.set_match_func(lambda completion, key, itr: key.lower() in self.completion.get_model()[itr][0].lower())
			self.set_completion(self.completion)
		def connect(self, event, callback):
			super().connect("activate", callback, None) #notify::selected
		def get_model(self) -> CompatStringListStore:
			return self.completion.get_model()
		def set_model(self, model: CompatStringListStore):
			self.completion.set_model(model)
		def get_selected_item(self) -> Optional[str]:
			selected = None
			iter = self.completion.get_model().get_iter_first()
			while selected == None and iter is not None:
				if self.get_text() == self.completion.get_model().get_value(iter, 0):
					selected = self.get_text()
				iter = self.completion.get_model().iter_next(iter)
			return selected
		def set_selected_item(self, value: str):
			self.set_text(value)
	class CompatCalendar(Gtk.Calendar):
		def select_day(self, gDate):
			self.select_month(gDate.get_month() - 1, gDate.get_year())
			super().select_day(gDate.get_day_of_month())
	class CompatClipBoard():
		def get() -> str:
			return Gtk.Clipboard.get(Gdk.SELECTION_CLIPBOARD).wait_for_text()
		def set(content: str):
			Gtk.Clipboard.get(Gdk.SELECTION_CLIPBOARD).set_text(content, -1)
	class CompatCssClass():
		def add_css_class(widget, classname):
			widget.get_style_context().add_class(classname)
else: #GTK4
	gi.require_version('Gtk', '4.0'); from gi.repository import GLib, Gtk, Gio, Gdk
	class CompatApplicationWindow(Gtk.ApplicationWindow):
		pass
	class CompatWindow(Gtk.Window):
		pass
	class CompatScrolledWindow(Gtk.ScrolledWindow):
		pass
	class CompatBox(Gtk.Box):
		pass
	class CompatButton(Gtk.Button):
		def new_from_icon_name(icon_name: str):
			return Gtk.Button.new_from_icon_name(icon_name)
	class CompatRadioButton(Gtk.CheckButton):
		pass
	class CompatAboutDialog(Gtk.AboutDialog):
		pass
	class CompatHeaderBar(Gtk.HeaderBar):
		pass
	class CompatPasswordEntry(Gtk.PasswordEntry):
		def __init__(self):
			super().__init__()
			self.set_show_peek_icon(True)
	class CompatStringListStore(Gio.ListStore):
		#Gio.ListStore.new(Gtk.StringObject)
		def append(self, value: str):
			super().append(Gtk.StringObject.new(value))
	class CompatDropDown(Gtk.DropDown):
		def __init__(self):
			super().__init__()
			self.set_enable_search(True)
			self.set_search_match_mode(Gtk.StringFilterMatchMode.SUBSTRING)
			self.set_expression(Gtk.PropertyExpression.new(Gtk.StringObject, None, 'string'))
		def get_selected_item(self) -> Optional[str]: #added method
			item = super().get_selected_item()
			return None if item == None else item.get_string()
		def set_selected_item(self, value: str): #added method
			found, idx = self.get_model().find_with_equal_func(Gtk.StringObject.new(value), lambda item, value, _: item.get_string() == value.get_string(), None)
			self.set_selected(idx if found else Gtk.INVALID_LIST_POSITION)
	class CompatCalendar(Gtk.Calendar):
		pass
	class CompatClipBoard():
		text = None
		def get_clipboard():
			return Gdk.Display.get_clipboard(Gdk.Display.get_default())
		def get() -> str: #sync !
			loop = GLib.MainLoop()
			CompatClipBoard.text = None
			def on_text_received(clipboard, task):
				CompatClipBoard.text = clipboard.read_text_finish(task)
				loop.quit()
			Gdk.Display.get_clipboard(Gdk.Display.get_default()).read_text_async(None, on_text_received)
			loop.run() #blocs until quit
			return CompatClipBoard.text
		def set(content: str):
			Gdk.Display.get_clipboard(Gdk.Display.get_default()).set(content)
	class CompatCssClass():
		def add_css_class(widget, classname):
			widget.add_css_class(classname)



class MarkDownTableFormatter(Gtk.Application):
	def __init__(self):
		super().__init__(application_id='info-fox-echo.markdown-text-formatter') #flags=Gio.ApplicationFlags.DEFAULT_FLAGS

	def do_activate(self):
		self.win = self.props.active_window
		if self.win == None:
			self.win = CompatApplicationWindow(application=self)
			self.win.set_default_size(800, 600)
			header_bar = CompatHeaderBar()
			header_bar.set_title_widget(Gtk.Label(label=_('Markdown Table Formatter')))
			about_button = CompatButton.new_from_icon_name('help-about')
			about_button.connect('clicked', self.show_about_dialog)
			header_bar.pack_start(about_button)
			self.win.set_titlebar(header_bar)
			self.box = CompatBox(orientation=Gtk.Orientation.VERTICAL, spacing=10)
			self.box.set_margin_top(10); self.box.set_margin_bottom(10); self.box.set_margin_start(10); self.box.set_margin_end(10)
			self.win.set_child(self.box)
			Gtk.CssProvider().load_from_data("textview.monospace { font-family: 'Monospace'; }".encode())
			self.add_widgets()
		self.win.present()


	def add_widgets(self):
		hbox = CompatBox(orientation=Gtk.Orientation.HORIZONTAL, spacing=10)
		hbox.append(Gtk.Label(label=_('Markdown table or CSV with <Tab> separator:')))
		label = Gtk.Label(); label.set_hexpand(True); hbox.append(label)
		paste_button = CompatButton.new_from_icon_name('edit-paste')
		paste_button.set_tooltip_text(_('Paste'))
		paste_button.connect('clicked', self.on_paste_button_clicked)
		hbox.append(paste_button)
		self.box.append(hbox)
		self.src_area = Gtk.TextView()
		#self.src_area.set_vexpand(True)
		CompatCssClass.add_css_class(self.src_area, 'monospace')
		scrolled_window = CompatScrolledWindow()
		scrolled_window.set_vexpand(True)
		scrolled_window.set_propagate_natural_height(True)
		scrolled_window.set_size_request(780, 240)
		scrolled_window.set_child(self.src_area)
		self.box.append(scrolled_window)
		self.on_src_area_changed_handler = self.src_area.get_buffer().connect('changed', self.on_src_area_changed)

		self.box.append(Gtk.Separator(orientation=Gtk.Orientation.HORIZONTAL))
		hbox = CompatBox(orientation=Gtk.Orientation.HORIZONTAL, spacing=10)
		hbox.append(Gtk.Label(label=_('Markdown table:')))
		label = Gtk.Label(); label.set_hexpand(True); hbox.append(label)
		copy_button = CompatButton.new_from_icon_name('edit-copy')
		copy_button.set_tooltip_text(_('Copy'))
		copy_button.connect('clicked', self.on_copy_button_clicked)
		hbox.append(copy_button)
		self.box.append(hbox)
		self.md_area = Gtk.TextView()
		CompatCssClass.add_css_class(self.md_area, 'monospace')
		#self.md_area.set_vexpand(True)
		scrolled_window = CompatScrolledWindow()
		scrolled_window.set_vexpand(True)
		scrolled_window.set_propagate_natural_height(True)
		scrolled_window.set_size_request(780, 240)
		scrolled_window.set_child(self.md_area)
		self.box.append(scrolled_window)

	def on_src_area_changed(self, textbuffer):
		start_iter, end_iter =  self.src_area.get_buffer().get_bounds()
		text =  self.src_area.get_buffer().get_text(start_iter, end_iter, True)
		if text.startswith('|'):
			text = convert_from_markdown_table(text)
			GLib.idle_add(self.idle_set_text, text)
		else:
			markdown = convert_to_markdown_table(text)
			if markdown:
				self.md_area.get_buffer().set_text(markdown)
	def idle_set_text(self, text):
		self.src_area.get_buffer().set_text(text)

	# ~ def on_paste_button_clicked(self, button):
		# ~ CompatClipBoard.get_clipboard().read_text_async(None, self.on_paste_text)
	# ~ def on_paste_text(self, clipboard, result):
		# ~ text = clipboard.read_text_finish(result)
		# ~ if text:
			# ~ self.src_area.get_buffer().set_text(text)
	def on_paste_button_clicked(self, button):
		text = CompatClipBoard.get()
		if text:
			self.src_area.get_buffer().set_text(text)

	def on_copy_button_clicked(self, button):
		start_iter, end_iter = self.md_area.get_buffer().get_bounds()
		text = self.md_area.get_buffer().get_text(start_iter, end_iter, True)
		if text:
			CompatClipBoard.set(text)

	def show_about_dialog(self, widget):
		about_dialog = CompatAboutDialog(transient_for=self.win, modal=True)
		about_dialog.set_program_name('Markdown Table Formatter')
		about_dialog.set_version('1.0')
		about_dialog.set_authors(['Frank ENDRES <first@last.fr>'])
		about_dialog.set_translator_credits('\n- de: Frank ENDRES\n- fr: Frank ENDRES')
		about_dialog.set_website('https://fox-echo.info/markdown-table-formatter')
		about_dialog.set_license('- CeCILL (Ce[a] C[nrs] I[nria] L[ogiciel] L[ibre])\n- GPL (General Public License)')
		about_dialog.set_logo_icon_name('markdown-table-formatter')
		about_dialog.set_visible(True)


def convert_to_markdown_table(text: str) -> str:
	table = [line.split('\t') for line in text.split('\n')]
	columns_count = max(len(line) for line in table)
	[line.extend([''] * (columns_count - len(line))) for line in table]
	max_widths = [max(len(line[col]) for line in table) for col in range(columns_count)]

	if all(column.startswith('**') and column.endswith('**') for column in table[0]):
		markdown = '| ' + ' | '.join([table[0][c].center(max_widths[c]) for c in range(len(table[0]))]) + ' |\n'
	else:
		markdown = '| ' + ' | '.join([table[0][c].ljust(max_widths[c]) for c in range(len(table[0]))]) + ' |\n'
	markdown += '| ' + ' | '.join(['-' * max_widths[c] for c in range(len(max_widths))]) + ' |\n'
	for l in range(1, len(table)):
		markdown += '| ' + ' | '.join([table[l][c].ljust(max_widths[c]) for c in range(len(table[l]))]) + ' |\n'
	return markdown[:-1] #remove trailing '\n'

def convert_from_markdown_table(text: str) -> str:
	lines = text.split('\n')
	del lines[1]
	source = ''
	for line in lines:
		source += '\t'.join([cell.strip(' ') for cell in line[1:-1].split('|')]) + '\n'
	return source[:-1] #remove trailing '\n'

def ctrl_c_handler(sig, frame):
	app.quit()

if __name__ == '__main__':
	# ~ gettext.textdomain('markdown-text-formatter')
	signal.signal(signal.SIGINT, ctrl_c_handler)
	app = MarkDownTableFormatter()
	app.run()
